<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\CopyTrade;
use App\Models\CopiedTrade;
use App\Models\Settings;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CopyTradeController extends Controller
{
    public function index()
{
    $settings = Settings::first();
    $user = auth()->user();
    
    // Calculate copy trade stats
    $copyTrades = CopyTrade::where('follower_id', $user->id)->get();
    $totalProfit = 0;
    $successfulTrades = 0;
    $totalTrades = 0;

    foreach ($copyTrades as $copyTrade) {
        $copiedTrades = CopiedTrade::with('copiedTrade')
            ->where('copy_trade_id', $copyTrade->id)
            ->where('status', 'completed')
            ->get();

        foreach ($copiedTrades as $trade) {
            $totalProfit += $trade->profit_amount ?? 0;
            if ($trade->result === 'win') {
                $successfulTrades++;
            }
            $totalTrades++;
        }
    }

    $successRate = $totalTrades > 0 ? ($successfulTrades / $totalTrades) * 100 : 0;
    
    // Get traders with their stats
    $traders = User::with(['trades', 'completedTrades', 'winningTrades', 'losingTrades'])
        ->whereNotNull('trader_type')
        ->where('id', '!=', auth()->id())
        ->where('is_trader', true)
        ->has('trades')
        ->get()
        ->map(function ($trader) {
            $trader->total_trades = $trader->trades->count();
            $trader->won_trades = $trader->winningTrades->count();
            $trader->lost_trades = $trader->losingTrades->count();
            
            // Calculate win rate
            $trader->win_rate = $trader->total_trades > 0 
                ? ($trader->won_trades / $trader->total_trades) * 100 
                : 0;
                
            return $trader;
        })
        ->sortByDesc('win_rate')
        ->values();

    // Paginate the collection
    $currentPage = request()->get('page', 1);
    $perPage = 12;
    $traders = new \Illuminate\Pagination\LengthAwarePaginator(
        $traders->forPage($currentPage, $perPage),
        $traders->count(),
        $perPage,
        $currentPage,
        ['path' => request()->url(), 'query' => request()->query()]
    );

    // Get currently copying traders
    $copyingTraders = CopyTrade::where('follower_id', auth()->id())
        ->where('status', 'active')
        ->pluck('trader_id')
        ->toArray();

    return view("{$settings->theme}.copy-trader.index", compact(
        'traders', 
        'copyingTraders',
        'totalProfit',
        'successRate',
        'totalTrades'
    ));
}

    public function startCopying(Request $request, User $trader)
    {
        $request->validate([
            'amount_per_trade' => 'required|numeric|min:1',
            'duration' => 'required|in:default,1m,5m,15m,30m,1h,4h,1d'
        ]);

        // Check if already copying
        if (CopyTrade::where('follower_id', auth()->id())
            ->where('trader_id', $trader->id)
            ->exists()) {
            return response()->json([
                'success' => false,
                'message' => 'You are already copying this trader'
            ], 422);
        }

        try {
            CopyTrade::create([
                'follower_id' => auth()->id(),
                'trader_id' => $trader->id,
                'amount_per_trade' => $request->amount_per_trade,
                'duration' => $request->duration,
                'status' => 'active'
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Successfully started copying trader'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to start copying trader'
            ], 500);
        }
    }

    public function stopCopying(User $trader)
    {
        try {
            CopyTrade::where('follower_id', auth()->id())
                ->where('trader_id', $trader->id)
                ->delete();

            return response()->json([
                'success' => true,
                'message' => 'Successfully stopped copying trader'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to stop copying trader'
            ], 500);
        }
    }
}