<?php

namespace App\Livewire\User\Kyc;

use App\Mail\UserSubmitKycMail;
use App\Models\Kyc;
use App\Models\Settings;
use App\Models\User;
use App\Notifications\User\KycApplicationReceivedNotification;
use App\Rules\PhoneNumberRule;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Mail;
use Illuminate\Validation\Rules\File;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Log;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class Form extends Component
{
    use WithFileUploads;
    use LivewireAlert;

    public $first_name;
    public $last_name;
    public $email;
    public $phone_number;
    public $dob;
    public $social_media;
    public $address;
    public $city;
    public $state;
    public $country;
    public $document_type = "Int'l Passport";
    public $frontImg;
    public $backImg;
    public $status = 'under review';

    public function mount()
    {
        $settings = Settings::select('id', 'enable_kyc')->find(1);
        if (! $settings->enable_kyc || auth()->user()->account_verify === 'under review') {
            session()->flash('message', 'Start KYC Application');
            return redirect()->route('user.kyc.start');
        }
    }

    public function render()
    {
        $template = Settings::select('id', 'theme')->find(1)->theme;
        return view("{$template}.kyc.form")
            ->extends("layouts.{$template}")
            ->title('Start KYC');
    }

    public function submit()
    {
        // Validate input
        $validated = $this->validate([
            'first_name' => ['required', 'string', 'max:190'],
            'last_name' => ['required', 'string', 'max:190'],
            'email' => ['required', 'email', 'max:190'],
            'phone_number' => ['required', new PhoneNumberRule()],
            'dob' => ['required', 'date'],
            'social_media' => ['nullable', 'string', 'max:190'],
            'address' => ['required', 'string', 'max:190'],
            'city' => ['required', 'string', 'max:190'],
            'state' => ['required', 'string', 'max:190'],
            'country' => ['required', 'string', 'max:190'],
            'document_type' => ['required', 'string', 'max:190'],
            'frontImg' => [
                'required',
                File::types(['jpg', 'jpeg', 'png', 'webp', 'pdf'])->max('30mb'),
            ],
            'backImg' => [
                'required',
                File::types(['jpg', 'jpeg', 'png', 'webp', 'pdf'])->max('30mb')
            ],
            'status' => ['required'],
        ]);

        try {
            // Get user
            $user = User::find(auth()->user()->id);

            // Store files temporarily to storage/public and then copy safely to public/assets/kyc
            $frontStored = $this->frontImg->store('kyc', 'public');
            $backStored = $this->backImg->store('kyc', 'public');

            $frontSourcePath = storage_path('app/public/' . $frontStored);
            $backSourcePath = storage_path('app/public/' . $backStored);
            $frontFileName = basename($frontStored);
            $backFileName = basename($backStored);

            $destinationDir = public_path('assets/kyc');
            if (!is_dir($destinationDir)) {
                if (!@mkdir($destinationDir, 0755, true) && !is_dir($destinationDir)) {
                    Log::error('Failed to create KYC directory', ['destinationDir' => $destinationDir]);
                    $this->alert(type: 'error', message: __('t.something_went_wrong'));
                    return;
                }
            }

            $frontDestinationPath = $destinationDir . DIRECTORY_SEPARATOR . $frontFileName;
            $backDestinationPath = $destinationDir . DIRECTORY_SEPARATOR . $backFileName;

            // Copy files to assets directory without emitting warnings
            if (!file_exists($frontSourcePath) || !@copy($frontSourcePath, $frontDestinationPath)) {
                Log::error('Failed to copy KYC front image', ['source' => $frontSourcePath, 'dest' => $frontDestinationPath]);
                $this->alert(type: 'error', message: __('t.something_went_wrong'));
                return;
            }
            if (!file_exists($backSourcePath) || !@copy($backSourcePath, $backDestinationPath)) {
                Log::error('Failed to copy KYC back image', ['source' => $backSourcePath, 'dest' => $backDestinationPath]);
                $this->alert(type: 'error', message: __('t.something_went_wrong'));
                return;
            }

            // Clean up temp files
            if (file_exists($frontSourcePath)) { @unlink($frontSourcePath); }
            if (file_exists($backSourcePath)) { @unlink($backSourcePath); }

            // Relative paths for DB
            $image_front = 'assets/kyc/' . $frontFileName;
            $image_back = 'assets/kyc/' . $backFileName;

            // Save to DB
            $filtered = Arr::except($validated, ['frontImg', 'backImg']);
            $kyc = Kyc::create($filtered + ['frontimg' => $image_front, 'backimg' => $image_back, 'user_id' => $user->id]);

            $settings = Settings::select(['id', 'receive_kyc_submission_email', 'send_kyc_status_email', 'notifiable_email'])->find(1);

            // Update user
            $user->account_verify = 'under review';
            $user->save();

           
            // Notify user and admin (do not fail the flow on mail errors)
            if ($settings->send_kyc_status_email) {
                $message = __('t.kyc_submit_email_sent');
                try {
                    $user->notify(new KycApplicationReceivedNotification(kyc: $kyc, message: $message));
                } catch (\Throwable $e) {
                    Log::error('KYC user notification failed', [
                        'user_id' => $user->id,
                        'kyc_id' => $kyc->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            if ($settings->receive_kyc_submission_email) {
                try {
                    Mail::to($settings->notifiable_email)->send(new UserSubmitKycMail(kyc: $kyc));
                } catch (\Throwable $e) {
                    Log::error('KYC admin email failed', [
                        'kyc_id' => $kyc->id,
                        'to' => $settings->notifiable_email,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            // Reset inputs
            $this->reset();

            // Show success and redirect (persisted alert)
            $this->flash(type: 'success', message: __('t.kyc_submit_success'), redirect: route('user.kyc.start'));
            return;

        } catch (\Throwable $e) {
            Log::error('KYC submit failed', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            $this->alert(type: 'error', message: __('t.kyc_submit_email_failed'));
        }

    }
}
