<?php

namespace App\Livewire\User\TradeSignal;

use App\Exceptions\SignalErrorException;
use App\Models\Settings;
use App\Models\Transaction;
use App\Models\User;
use App\Services\SignalService;
use Illuminate\Support\Collection;
use Livewire\Attributes\Computed;
use Livewire\Component;

class Index extends Component
{
    public $set;
    public $page;
    public $nextPageUrl;
    public $previousPageUrl;
    public Collection $signals;
    public ?array $subscription = null;

    public function mount(SignalService $signal, string $page)
    {
        $this->page = $page;
        $this->signals = collect([]); // Initialize as empty collection

        // Check if signals module is enabled
        $settings = Settings::select('modules')->find(1);
        abort_if($settings->modules['signal'] !== 'true', 404);

        try {
            $this->set = $signal->settings();
        } catch (SignalErrorException $e) {
            session()->flash('warning', $e->getMessage());
            return redirect()->route('user.dashboard');
        } catch (\Throwable $e) {
            session()->flash('warning', $e->getMessage());
            return redirect()->route('user.dashboard');
        }
    }

    #[Computed]
    public function signals(SignalService $signal)
    {
        try {
            $data = $signal->signals($this->page);
            $this->nextPageUrl = $data['next_page_url'];
            $this->previousPageUrl = $data['prev_page_url'];
            return collect($data['data']); // Return as collection
        } catch (SignalErrorException) {
            return collect([]); // Return empty collection
        } catch (\Throwable) {
            return collect([]); // Return empty collection
        }
    }

    public function render(SignalService $signal)
    {
        try {
            // Get subscription data
            $this->subscription = $signal->subscription(auth()->user()->id);

            // Get signals data
            $data = $signal->signals($this->page);
            
            // Convert to collection and filter
            $this->signals = collect($data['data'])
                ->where('status', 'published')
                ->values(); // Use values() to reindex array keys

            // Set pagination URLs
            $this->nextPageUrl = $data['next_page_url'];
            $this->previousPageUrl = $data['prev_page_url'];

        } catch (SignalErrorException $e) {
            $this->signals = collect([]); // Reset to empty collection
            session()->flash('message', $e->getMessage());
        } catch (\Throwable $e) {
            $this->signals = collect([]); // Reset to empty collection
            session()->flash('message', $e->getMessage());
        }

        $template = Settings::select('theme')->find(1)->theme;
        
        return view("{$template}.trade-signal.index", [
            'subscription' => $this->subscription,
        ])
            ->extends("layouts.{$template}")
            ->title('Trade signals');
    }

   public function renew(SignalService $signal, string $fee)
{
    try {
        $user = User::findOrFail(auth()->user()->id);
        $amount = floatval($fee);

        // Check balance
        if ($user->account_bal < $amount) {
            session()->flash('message', __('t.renewal_insufficient_funds'));
            return;
        }

        // Process renewal
        $signal->renew(['id' => $user->id]);

        // Update user balance
        $user->account_bal -= $amount;
        $user->save();

        // Create transaction record
        Transaction::create([
            'user_id' => $user->id,
            'amount' => $amount,
            'narration' => __('t.signal_subscription_narration'),
            'type' => 'Debit',
        ]);

        session()->flash('success', __('t.renewal_success'));
        
    } catch (SignalErrorException $e) {
       session()->flash('message', $e->getMessage());
    } catch (\Throwable $e) {
        session()->flash('message', __('t.renewal_error'));
    }
}


    public function placeholder(array $params = [])
    {
        return view('livewire.placeholders.skeleton', $params);
    }
}